% Copyright (c) 2016 Giampaolo D'Alessandro
% Mathematical Sciences - University of Southampton
% Email: dales@soton.ac.uk
% 
% Permission is hereby granted, free of charge, to any person obtaining a copy
% of this software and associated documentation files (the "Software"), to deal
% in the Software without restriction, including without limitation the rights
% to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
% copies of the Software, and to permit persons to whom the Software is
% furnished to do so, subject to the following conditions:
% 
% The above copyright notice and this permission notice shall be included in
% all copies or substantial portions of the Software.
% 
% THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
% IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
% FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
% AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
% LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
% OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
% SOFTWARE.

% Script to thermal nonlinearity of a liquid crystal cell
% illuminated by a Gaussian beam with periodic modulation in the
% x-direction.
%
% Input parameters
% I_0 (scalar) - The input power intensity in W cm^{-2}
% V_dim (scalar) - The voltage applied to the cell in Volts

% 0) Set up parallel processing (useful for the thermal code)
% parobj = parpool;

% 1) Dimensional parameters

% 1.1) Cell parameters
L_LC = 50e-6;     % Cell thickness in metres
Lambda = 50e-6;   % Grating period in metres
L_G_dim = 5e-4;   % Glass thickness in metres

% 1.2) Thermal parameters - E7 data
% Wavelength for which data is available.
lambda_m = [450; 486; 546; 589; 633; 656]*1e-9; % in m
% Absorbance data - Here too we perform a spline interpolation.
alpha_m =  [165 139 110 91 75 70]*1e2;  % in m^{-1}
% Maximum absorbance coefficient in m^{-1}
alpha_interp = griddedInterpolant(lambda_m,alpha_m,'spline');
sigma_pump_dim = 300e-6;   % Pump beam half-width in metres
kappa_LC = 0.15;  % LC thermal conductivity in W K^{-1} m^{-1} [also 0.19]
kappa_G = 1;   % Glass thermal conductivity in W K^{-1} m^{-1} [Google]
I_0 = 19.5*1e4;     % Power of the input beam in W m^{-2}
lambda_pump_dim = 532e-9;  % Pump wavelength in m

% 1.3) Domain size parameters
Lx_dim = 8*sigma_pump_dim;  % Width of the cell transverse plane (x-direction) 
Ly_dim = 8*sigma_pump_dim;  % Width of the cell transverse plane (x-direction) 

% 1.4) Alignment parameters

% 1.4.1) Elastic terms
K = [11 10 17]*1e-12; %Elastic constants in Newtons
theta_L = degtorad(1.4);  % Pretilt angle at $z=-d$ (dimensional units).
theta_R = degtorad(1.4);  % Pretilt angle at $z=+d$ (dimensional units).

% 1.4.2) Electrostatic coupling parameters
eps_0 = 8.85e-12;   % Dielectric constant of vacuum.
eps_par = 19;    % Extraordinary AC dielectric permittivity
eps_perp = 5.2;    % Ordinary AC dielectric permittivity
V_dim = 0;  % Voltage applied to the cell in Volts

% 1.5) Diffraction parameter
sigma_probe_dim = 300e-6;  % Probe beam half-width (nanometers)
lambda_probe_dim = 633e-9;   % Probe wavelength in nanometers
Probe_beam_angleInDegrees = 15;
Probe_beam_angle = degtorad(Probe_beam_angleInDegrees);

% 2) Numerical parameters

% Specify the number of grid points in x and y directions (there should be
% enough points to identify clearly the period of the modulation or the
% width of the beam, whichever is smallest.
nx = 512;
ny = 512;
% Specify the number of points in the z-direction (i.e. into the cell).
nz = 16;
% Numver of voltage values
nV = length(V_dim);

ngppp = round(Lambda/(Lx_dim/(nx+1)));
display(sprintf('Number of grid points per period: %d',ngppp));
if (ngppp<4)
  error('TempProf:ngppp',...
    'The number of grid points per period is too small');
end

% 3) Plot parameters

FontName='Times';
FontSize=18;
FontAngle='normal'; %'italic';
LineWidth=2;
MarkerSize=6;  % In points (72 points per inch)

% 4) Compute the non-dimensional parameters

% 4.1) Geometrical parameters
d = 0.5*L_LC;     % Length scaling factor
L_G = L_G_dim/d;  % Glass thickness
Lx = Lx_dim/d;    % Width of cell in the x-direction
Ly = Ly_dim/d;    % Width of cell in the y-direction

% 4.2) Thermal parameters
T_0 = d*I_0/kappa_LC;      % Temperature scaling coefficient
kappa = kappa_G/kappa_LC;  % Ratio of the termal conductivities
sigma_pump = sigma_pump_dim/d;      % Beam half-width
q = 2*pi/(Lambda/d);       % Grating wavenumber

% 4.3) Coefficients in the liquid crystal equation

% Electrostatic potential scaling coefficient
V_0 = sqrt((K(1)+K(3))/eps_0);
V = V_dim/V_0;  % Nondimensional cell potential
% Liquid crystal coupling coefficient in electrostatic equation
alpha_psi = (eps_par-eps_perp)/(eps_par+eps_perp);
% Electrostatic coupling coefficient in liquid crystal equation
delta_1 = eps_par-eps_perp; 
% Normalised bend--splay differernce
delta_2 = (K(3)-K(1))/(K(3)+K(1));

% 4.4) z-coordinate and quadrature weights
% We must define ithe coordinate here to ensure that the director
% field and the temperature are computed at the same points.
% [~,z] = cheb(nz-1);
[z,W,P] = DefClencurt(nz-1);

% 4.5) Diffraction parameters
sigma_p = sigma_probe_dim/d;

%% Compute and plot the director profile
% 
[theta] = DirectorProfile(V,delta_1,delta_2,alpha_psi,...
                          theta_L,theta_R,z);

% Plot the director profile

% Uniform grid for plotting
z_uni = linspace(0,L_LC);
% Color coding
col = 'kbcgmr';
figure(1); clf;
set(gcf,'PaperOrientation','Landscape','PaperType','A4',...
  'PaperUnits','normalized','PaperPosition',[0 0 1 1]);
axes('Parent',gcf,...
  'YMinorTick','on','XMinorTick','on',...
  'FontSize',FontSize,'FontName',FontName,'FontAngle',FontAngle);
hold(gca,'all');
% Deletes the legend array if it has been created.  This is needed to avoid
% an error message if the number of voltage values is decreased between
% successive runs and the memory stack has not been cleared.
clear gr_pl;
for k = 1:nV
  % Define an interpolating function to be used for the plot
  Theta_int = griddedInterpolant(...
    (z(nz:-1:1)-min(z))/(max(z)-min(z))*L_LC,theta(nz:-1:1,k),'spline');
  gr_pl(k) = hggroup('DisplayName',sprintf('V=%.1f',V_dim(k))); %#ok<SAGROW>
  plot((z-min(z))/(max(z)-min(z))*L_LC,theta(:,k)/(0.5*pi),'or', ...
    'MarkerSize',MarkerSize,'Parent',gr_pl(k));
  plot(z_uni,Theta_int(z_uni)/(0.5*pi),...
    'LineStyle','-','LineWidth',LineWidth,'Color',col(1+mod(k-1,6)), ...
    'Parent',gr_pl(k));
end
xlabel('z');
ylabel('\theta(z) \times \pi/2');
axis([0,L_LC,0,max(theta(:))/(0.5*pi)*1.05]);
title('Director profile');
legend(gr_pl,'Location','EastOutside');

%% Compute and plot the temperature profile

% This code assumes that the temperature profile is independent of the
% orientation of the liquid crystal. We are neglecting:
% 1) The thermal anisotropy of the liquid crystal
% 2) The shift in resonance and ,hence, the change of absorption
% coefficient of the nanoparticles caused by the change of effective
% refractive index. 
% This second approximation may be justifiable if the particles are
% spherical, but fails in the case of nanorods.

% Define the non-dimensional absorption functions
% (*) Pump absorption
% Maximum pump absorption
alpha_pump_dim = alpha_interp(lambda_pump_dim);
% This absorption profile is uniform: it is suitable if the absorption of
% the nanoparticles is not sensitive to the alignment of the liquid
% crystal.
alpha_pump = @(z) alpha_pump_dim*d*ones(size(z));
% (*) Probe absorption
% Maximum probe absorption
alpha_probe_dim = alpha_interp(lambda_probe_dim);
% Total absorption in propagattion assuming that the absorption is uniform.
alpha_probe = @(z) alpha_probe_dim*d*ones(size(z))/cos(Probe_beam_angle);
alpha_probe_tot = W(1,:)*alpha_probe(z);

[x,y,T] = ThermalProfile(L_G,Lx,Ly,nx,ny,z,alpha_pump,kappa,sigma_pump,q,W);
[X,Y,Z] = meshgrid(x,y,z);

%% Plot the temperature profile as a contour slice

figure(2); clf;
set(gcf,'PaperOrientation','Landscape','PaperType','A4',...
  'PaperUnits','normalized','PaperPosition',[0 0 1 1]);
axes('Parent',gcf,...
  'YMinorTick','on','XMinorTick','on',...
  'FontSize',FontSize,'FontName',FontName,'FontAngle',FontAngle);
view(gca,[-44.5 16]);
box(gca,'on');
grid(gca,'on');
hold(gca,'all');
% h = slice(X*d,Y*d,Z*d,T_0*T,[],[],d*z(round(linspace(1,nz,4))));
h = slice(X*d*1e3,Y*d*1e3,Z*d*1e6,T,[],0,1e6*linspace(-d,d,4)); %d*z(round(linspace(1,nz,4))));
set(h,'FaceColor','interp','EdgeColor','none','DiffuseStrength',.8);
xlabel('x [mm]'); ylabel('y[mm]'); zlabel('z[\mum]');
axis('tight');
title(sprintf(...
  'Temperature profile in the cell at I_{0}=%.1f W cm^{-2}',I_0*1e-4));
c_T = colorbar('FontSize',FontSize);
ylabel(c_T,'\Delta T [K]');
% The instruction below is needed to avoid a bug in the NVidia driver for
% Ubuntu: without it some colours appear white.
set(gcf,'Renderer','Zbuffer');

%% Compute the diffraction efficiency

% If the diffraction efficiency is computed for more than one voltage
% disable producing the figures.
allow_fig = (nV==1);

DiffEff = zeros(1,nV);
for k = 1:nV
  DiffEff(k) = RamanNath(lambda_probe_dim,x*d,y*d,z*d,T*T_0,theta(:,k),...
    alpha_probe_tot,q/d,sigma_probe_dim,allow_fig);
end

% Plot the diffraction efficiency
figure(8); clf;
set(gcf,'PaperOrientation','Landscape','PaperType','A4',...
  'PaperUnits','normalized','PaperPosition',[0 0 1 1]);
axes('Parent',gcf,...
  'YMinorTick','on','XMinorTick','on',...
  'FontSize',FontSize,'FontName',FontName,'FontAngle',FontAngle);
hold(gca,'all');
plot(V_dim,DiffEff,'b-o','LineWidth',LineWidth,'MarkerSize',MarkerSize);
xlabel('Voltage [Volt]');
ylabel('Diffraction Efficiency');

% Close the parallel pool
% delete(gcp);